//+------------------------------------------------------------------+
//|                                      Heikin_Ashi_Acc_Optimized.mq5|
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
//| Version optimisée - Réduction mémoire significative              |
//| - Calcul incrémental (pas de recalcul total)                     |
//| - Tableaux MTF à taille fixe limitée                             |
//| - Suppression des buffers inutilisés                             |
//| - Recherche binaire pour mapping MTF                             |
//+------------------------------------------------------------------+
#property indicator_separate_window
#property indicator_buffers 7
#property indicator_plots   6

//--- Plots
#property indicator_label1  "Cumul Positif MTF"
#property indicator_type1   DRAW_HISTOGRAM
#property indicator_color1  clrSpringGreen
#property indicator_style1  STYLE_SOLID
#property indicator_width1  5

#property indicator_label2  "Cumul Négatif MTF"
#property indicator_type2   DRAW_HISTOGRAM
#property indicator_color2  clrSalmon
#property indicator_style2  STYLE_SOLID
#property indicator_width2  5

#property indicator_label3  "Différence Absolue MTF"
#property indicator_type3   DRAW_LINE
#property indicator_color3  clrDeepSkyBlue
#property indicator_style3  STYLE_SOLID
#property indicator_width3  3

#property indicator_label4  "MA Abs Diff MTF"
#property indicator_type4   DRAW_LINE
#property indicator_color4  clrForestGreen
#property indicator_style4  STYLE_SOLID
#property indicator_width4  2

#property indicator_label5  "-MA Abs Diff MTF"
#property indicator_type5   DRAW_LINE
#property indicator_color5  clrCrimson
#property indicator_style5  STYLE_DOT
#property indicator_width5  2

#property indicator_label6  "MA Diff MTF"
#property indicator_type6   DRAW_LINE
#property indicator_color6  clrOrange
#property indicator_style6  STYLE_SOLID
#property indicator_width6  2

//--- Niveaux
#property indicator_levelcolor clrSilver
#property indicator_levelstyle STYLE_DOT
#property indicator_levelwidth 1
#property indicator_level1 25
#property indicator_level2 -25

//--- Paramètres MTF
input group "=== Configuration Multi-TimeFrame ==="
input ENUM_TIMEFRAMES InpTimeFrame = PERIOD_CURRENT;

//--- Paramètres de base
input group "=== Configuration Série ==="
input int InpOppositeBarCount = 4;
input int InpDrawdownPips = 40;
input bool InpUseDynamicDrawdown = true;

input group "=== Configuration Filtrage Pourcentage ==="
input int InpPercentagePeriod = 36;
input double InpPercentageThreshold = 40.0;

input group "=== Configuration Moyenne Mobile Différence ==="
input int InpMAPeriod = 7;
input ENUM_MA_METHOD InpMAMethod = MODE_EMA;

input group "=== Configuration Moyenne Mobile Valeur Absolue ==="
input int InpMAAbsPeriod = 14;
input ENUM_MA_METHOD InpMAAbsMethod = MODE_SMA;

input group "=== Configuration ATR (pour seuil dynamique) ==="
input int InpATRPeriod = 14;
input double InpATRMultiplier = 2.0;

input group "=== Configuration Horaires ==="
input bool InpEnableTimeFilter = true;    // Activer le filtre horaire
input int InpStartHour = 2;
input int InpStartMinute = 30;
input int InpEndHour = 22;
input int InpEndMinute = 30;

input group "=== Lissage MTF ==="
input bool InpEnableMTFSmoothing = true;  // Lisser les buffers en MTF
input double InpSmoothingFactor = 1.5;    // Facteur de lissage (1.0-3.0)

input group "=== Alertes ==="
input bool InpEnableAlerts = false;
input bool InpAlertOnSeriesStart = false;
input bool InpAlertOnSeriesStop = false;
input bool InpAlertOnCrossover = false;

input group "=== Affichage ==="
input bool InpDebugMode = false;

//--- Buffers principaux (7 au lieu de 8)
double PositiveBuffer[];
double NegativeBuffer[];
double DifferenceBuffer[];
double MADifferenceBuffer[];
double MAAbsDifferenceBuffer[];
double InverseMAAbsDifferenceBuffer[];
double AbsDifferenceBuffer[];

//--- Variables globales
ENUM_TIMEFRAMES g_timeframe;
int g_mtf_ratio = 1;
double g_point_multiplier = 1.0;
int g_atr_handle = INVALID_HANDLE;
int g_smoothing_period = 1;  // Période de lissage calculée

//--- État des séries (structure compacte)
struct SeriesState
{
   bool   active;
   int    opposite_count;
   double cumul;
   double extremum;
   
   void Reset()
   {
      active = false;
      opposite_count = 0;
      cumul = 0.0;
      extremum = 0.0;
   }
};

SeriesState g_positive_series;
SeriesState g_negative_series;

//--- Cache MTF dynamique
struct MTFBar
{
   datetime time;
   double   ha_open;
   double   ha_close;
   double   positive;
   double   negative;
   double   difference;
   double   abs_diff;
   double   ma_diff;
   double   ma_abs;
};

MTFBar g_mtf_cache[];
int    g_mtf_cache_count = 0;
datetime g_mtf_last_time = 0;
int g_mtf_last_bars_count = 0;

//--- Variables pour calcul incrémental
bool g_last_was_in_trading_hours = false;

//+------------------------------------------------------------------+
//| Initialization                                                   |
//+------------------------------------------------------------------+
int OnInit()
{
   Print("=== HEIKIN ASHI CUMULATOR MTF v6.02 OPTIMIZED ===");

   g_timeframe = (InpTimeFrame == PERIOD_CURRENT) ? _Period : InpTimeFrame;
   
   if(g_timeframe < _Period)
   {
      Alert("ERREUR: Le timeframe sélectionné ne peut pas être inférieur au graphique!");
      return INIT_PARAMETERS_INCORRECT;
   }

   g_mtf_ratio = PeriodSeconds(g_timeframe) / PeriodSeconds(_Period);
   if(g_mtf_ratio < 1) g_mtf_ratio = 1;

   // Calcul de la période de lissage MTF
   if(InpEnableMTFSmoothing && g_mtf_ratio > 1)
   {
      g_smoothing_period = (int)MathCeil(InpSmoothingFactor * g_mtf_ratio);
      if(g_smoothing_period < 1) g_smoothing_period = 1;
   }
   else
   {
      g_smoothing_period = 1;
   }

   // Validation des paramètres
   if(InpOppositeBarCount < 1 || InpOppositeBarCount > 10 ||
      InpDrawdownPips < 5 || InpDrawdownPips > 100 ||
      InpPercentagePeriod < 10 || InpPercentagePeriod > 200 ||
      InpPercentageThreshold < 1.0 || InpPercentageThreshold > 50.0 ||
      InpMAPeriod < 1 || InpMAPeriod > 200 ||
      InpMAAbsPeriod < 1 || InpMAAbsPeriod > 200 ||
      InpATRPeriod < 1 || InpATRPeriod > 100 ||
      InpATRMultiplier < 1.0 || InpATRMultiplier > 5.0 ||
      InpSmoothingFactor < 1.0 || InpSmoothingFactor > 3.0)
   {
      Print("ERREUR: Paramètres invalides");
      return INIT_PARAMETERS_INCORRECT;
   }

   if(InpEnableTimeFilter)
   {
      if(InpStartHour < 0 || InpStartHour > 23 ||
         InpEndHour < 0 || InpEndHour > 23 ||
         InpStartMinute < 0 || InpStartMinute > 59 ||
         InpEndMinute < 0 || InpEndMinute > 59)
      {
         Print("ERREUR: Paramètres horaires invalides");
         return INIT_PARAMETERS_INCORRECT;
      }
   }

   // ATR handle
   if(InpUseDynamicDrawdown)
   {
      g_atr_handle = iATR(_Symbol, g_timeframe, InpATRPeriod);
      if(g_atr_handle == INVALID_HANDLE)
      {
         Print("ERREUR: Impossible de créer l'indicateur ATR");
         return INIT_FAILED;
      }
   }

   // Configuration des buffers (7 au lieu de 8)
   SetIndexBuffer(0, PositiveBuffer, INDICATOR_DATA);
   SetIndexBuffer(1, NegativeBuffer, INDICATOR_DATA);
   SetIndexBuffer(2, DifferenceBuffer, INDICATOR_DATA);
   SetIndexBuffer(3, MAAbsDifferenceBuffer, INDICATOR_DATA);
   SetIndexBuffer(4, InverseMAAbsDifferenceBuffer, INDICATOR_DATA);
   SetIndexBuffer(5, MADifferenceBuffer, INDICATOR_DATA);
   SetIndexBuffer(6, AbsDifferenceBuffer, INDICATOR_CALCULATIONS);

   // Point multiplier
   int digits = (int)SymbolInfoInteger(_Symbol, SYMBOL_DIGITS);
   g_point_multiplier = (digits == 5 || digits == 3) ? 10.0 : 1.0;

   // Configuration des labels
   string tf_str = (g_timeframe == _Period) ? "Current" : EnumToString(g_timeframe);
   string smooth_str = (g_smoothing_period > 1) ? StringFormat(" S%d", g_smoothing_period) : "";
   string time_str = InpEnableTimeFilter ? StringFormat(" %02d:%02d-%02d:%02d", 
                     InpStartHour, InpStartMinute, InpEndHour, InpEndMinute) : "";
   
   IndicatorSetString(INDICATOR_SHORTNAME, 
      StringFormat("Heikin MTF OPT [%s] %dB/%dP%s%s", tf_str, InpOppositeBarCount, 
                   InpDrawdownPips, smooth_str, time_str));

   ResetAll();

   Print("Initialisé - TF: ", tf_str, " | Ratio: ", g_mtf_ratio, 
         " | Lissage: ", g_smoothing_period,
         " | Filtre horaire: ", InpEnableTimeFilter ? "ON" : "OFF");

   return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Reset complet                                                    |
//+------------------------------------------------------------------+
void ResetAll()
{
   g_positive_series.Reset();
   g_negative_series.Reset();
   g_last_was_in_trading_hours = false;
   g_mtf_cache_count = 0;
   g_mtf_last_time = 0;
   g_mtf_last_bars_count = 0;
}

//+------------------------------------------------------------------+
//| Vérification plage horaire                                       |
//+------------------------------------------------------------------+
bool IsInTradingHours(datetime bar_time)
{
   // Si le filtre horaire est désactivé, toujours dans les heures de trading
   if(!InpEnableTimeFilter)
      return true;
      
   MqlDateTime dt;
   TimeToStruct(bar_time, dt);

   int current_minutes = dt.hour * 60 + dt.min;
   int start_minutes = InpStartHour * 60 + InpStartMinute;
   int end_minutes = InpEndHour * 60 + InpEndMinute;

   if(start_minutes <= end_minutes)
      return (current_minutes >= start_minutes && current_minutes <= end_minutes);
   else
      return (current_minutes >= start_minutes || current_minutes <= end_minutes);
}

//+------------------------------------------------------------------+
//| Obtenir le seuil dynamique                                       |
//+------------------------------------------------------------------+
double GetDynamicDrawdown(int mtf_bar_index)
{
   if(!InpUseDynamicDrawdown || g_atr_handle == INVALID_HANDLE)
      return InpDrawdownPips;

   double atr_values[1];
   if(CopyBuffer(g_atr_handle, 0, mtf_bar_index, 1, atr_values) <= 0)
      return InpDrawdownPips;

   double atr_value = atr_values[0];
   if(atr_value <= 0 || atr_value != atr_value)
      return InpDrawdownPips;

   double atr_pips = atr_value / (_Point * g_point_multiplier);
   if(atr_pips <= 0 || atr_pips > 10000)
      return InpDrawdownPips;
   
   double dynamic_threshold = atr_pips * InpATRMultiplier;
   return MathMax(dynamic_threshold, InpDrawdownPips * 0.5);
}

//+------------------------------------------------------------------+
//| Traitement série positive                                        |
//+------------------------------------------------------------------+
void ProcessPositiveSeries(bool candle_positive, double pips, double threshold)
{
   if(candle_positive)
   {
      if(!g_positive_series.active)
      {
         g_positive_series.active = true;
         g_positive_series.cumul = pips;
         g_positive_series.extremum = pips;
         g_positive_series.opposite_count = 0;
      }
      else
      {
         g_positive_series.cumul += pips;
         if(g_positive_series.cumul > g_positive_series.extremum)
            g_positive_series.extremum = g_positive_series.cumul;
         g_positive_series.opposite_count = 0;
      }
   }
   else if(g_positive_series.active)
   {
      g_positive_series.opposite_count++;
      g_positive_series.cumul += pips;
      
      double drawdown = g_positive_series.extremum - g_positive_series.cumul;
      
      if(g_positive_series.opposite_count >= InpOppositeBarCount || drawdown >= threshold)
         g_positive_series.Reset();
   }
}

//+------------------------------------------------------------------+
//| Traitement série négative                                        |
//+------------------------------------------------------------------+
void ProcessNegativeSeries(bool candle_positive, double pips, double threshold)
{
   if(!candle_positive)
   {
      if(!g_negative_series.active)
      {
         g_negative_series.active = true;
         g_negative_series.cumul = pips;
         g_negative_series.extremum = pips;
         g_negative_series.opposite_count = 0;
      }
      else
      {
         g_negative_series.cumul += pips;
         if(g_negative_series.cumul < g_negative_series.extremum)
            g_negative_series.extremum = g_negative_series.cumul;
         g_negative_series.opposite_count = 0;
      }
   }
   else if(g_negative_series.active)
   {
      g_negative_series.opposite_count++;
      g_negative_series.cumul += pips;
      
      double drawdown = g_negative_series.cumul - g_negative_series.extremum;
      
      if(g_negative_series.opposite_count >= InpOppositeBarCount || drawdown >= threshold)
         g_negative_series.Reset();
   }
}

//+------------------------------------------------------------------+
//| Reconstruire le cache MTF complet                                |
//+------------------------------------------------------------------+
bool RebuildMTFCache()
{
   // Obtenir les données MTF
   datetime mtf_times[];
   double mtf_open[], mtf_high[], mtf_low[], mtf_close[];
   
   ArraySetAsSeries(mtf_times, true);
   ArraySetAsSeries(mtf_open, true);
   ArraySetAsSeries(mtf_high, true);
   ArraySetAsSeries(mtf_low, true);
   ArraySetAsSeries(mtf_close, true);
   
   int bars_available = iBars(_Symbol, g_timeframe);
   int bars_to_copy = bars_available;
   
   if(CopyTime(_Symbol, g_timeframe, 0, bars_to_copy, mtf_times) <= 0) return false;
   if(CopyOpen(_Symbol, g_timeframe, 0, bars_to_copy, mtf_open) <= 0) return false;
   if(CopyHigh(_Symbol, g_timeframe, 0, bars_to_copy, mtf_high) <= 0) return false;
   if(CopyLow(_Symbol, g_timeframe, 0, bars_to_copy, mtf_low) <= 0) return false;
   if(CopyClose(_Symbol, g_timeframe, 0, bars_to_copy, mtf_close) <= 0) return false;
   
   int copied = ArraySize(mtf_times);
   if(copied < 2) return false;
   
   // Redimensionner le cache dynamiquement
   ArrayResize(g_mtf_cache, copied);
   
   // Reset des séries
   g_positive_series.Reset();
   g_negative_series.Reset();
   g_last_was_in_trading_hours = false;
   
   // Remplir le cache du plus ancien au plus récent
   g_mtf_cache_count = 0;
   
   double prev_ha_open = 0.0;
   double prev_ha_close = 0.0;
   
   for(int i = copied - 1; i >= 0; i--)
   {
      int cache_idx = g_mtf_cache_count;
      
      // Calcul Heikin Ashi
      double ha_close = (mtf_open[i] + mtf_high[i] + mtf_low[i] + mtf_close[i]) / 4.0;
      double ha_open;
      
      if(cache_idx == 0)
         ha_open = (mtf_open[i] + mtf_close[i]) / 2.0;
      else
         ha_open = (prev_ha_open + prev_ha_close) / 2.0;
      
      // Sauvegarde pour la prochaine itération
      prev_ha_open = ha_open;
      prev_ha_close = ha_close;
      
      // Gestion des heures de trading
      bool is_in_hours = IsInTradingHours(mtf_times[i]);
      
      if(g_last_was_in_trading_hours && !is_in_hours)
      {
         g_positive_series.Reset();
         g_negative_series.Reset();
      }
      
      if(is_in_hours)
      {
         bool candle_positive = (ha_close > ha_open);
         double current_pips = (ha_close - ha_open) / (_Point * g_point_multiplier);
         double drawdown_threshold = GetDynamicDrawdown(i);
         
         ProcessPositiveSeries(candle_positive, current_pips, drawdown_threshold);
         ProcessNegativeSeries(candle_positive, current_pips, drawdown_threshold);
      }
      
      g_last_was_in_trading_hours = is_in_hours;
      
      // Stocker dans le cache
      g_mtf_cache[cache_idx].time = mtf_times[i];
      g_mtf_cache[cache_idx].ha_open = ha_open;
      g_mtf_cache[cache_idx].ha_close = ha_close;
      g_mtf_cache[cache_idx].positive = g_positive_series.active ? g_positive_series.cumul : 0.0;
      g_mtf_cache[cache_idx].negative = g_negative_series.active ? g_negative_series.cumul : 0.0;
      g_mtf_cache[cache_idx].difference = MathAbs(g_mtf_cache[cache_idx].positive) - 
                                          MathAbs(g_mtf_cache[cache_idx].negative);
      g_mtf_cache[cache_idx].abs_diff = MathAbs(g_mtf_cache[cache_idx].difference);
      g_mtf_cache[cache_idx].ma_diff = 0.0;
      g_mtf_cache[cache_idx].ma_abs = 0.0;
      
      g_mtf_cache_count++;
   }
   
   // Calculer les MA
   CalculateMAsOnCache();
   
   // Sauvegarder l'état
   if(copied > 0)
   {
      g_mtf_last_time = mtf_times[0];
      g_mtf_last_bars_count = bars_available;
   }
   
   return true;
}

//+------------------------------------------------------------------+
//| Mettre à jour la dernière barre du cache                         |
//+------------------------------------------------------------------+
bool UpdateLastMTFBar()
{
   if(g_mtf_cache_count == 0) return false;
   
   datetime mtf_times[1];
   double mtf_open[1], mtf_high[1], mtf_low[1], mtf_close[1];
   
   if(CopyTime(_Symbol, g_timeframe, 0, 1, mtf_times) <= 0) return false;
   if(CopyOpen(_Symbol, g_timeframe, 0, 1, mtf_open) <= 0) return false;
   if(CopyHigh(_Symbol, g_timeframe, 0, 1, mtf_high) <= 0) return false;
   if(CopyLow(_Symbol, g_timeframe, 0, 1, mtf_low) <= 0) return false;
   if(CopyClose(_Symbol, g_timeframe, 0, 1, mtf_close) <= 0) return false;
   
   int last_idx = g_mtf_cache_count - 1;
   
   // Vérifier si c'est la même barre
   if(g_mtf_cache[last_idx].time != mtf_times[0])
      return false; // Nouvelle barre, besoin de rebuild
   
   // Mettre à jour le Heikin Ashi
   double ha_close = (mtf_open[0] + mtf_high[0] + mtf_low[0] + mtf_close[0]) / 4.0;
   
   // Recalculer la série pour cette barre seulement
   // On doit restaurer l'état de la série avant cette barre
   // Pour simplifier, on ne recalcule que si la direction change
   
   bool prev_candle_positive = (g_mtf_cache[last_idx].ha_close > g_mtf_cache[last_idx].ha_open);
   bool curr_candle_positive = (ha_close > g_mtf_cache[last_idx].ha_open);
   
   // Si la direction n'a pas changé, mise à jour simple
   if(prev_candle_positive == curr_candle_positive)
   {
      g_mtf_cache[last_idx].ha_close = ha_close;
      
      double current_pips = (ha_close - g_mtf_cache[last_idx].ha_open) / (_Point * g_point_multiplier);
      
      // Mise à jour approximative des cumuls
      if(g_positive_series.active && curr_candle_positive)
      {
         g_mtf_cache[last_idx].positive = g_positive_series.cumul;
      }
      if(g_negative_series.active && !curr_candle_positive)
      {
         g_mtf_cache[last_idx].negative = g_negative_series.cumul;
      }
      
      g_mtf_cache[last_idx].difference = MathAbs(g_mtf_cache[last_idx].positive) - 
                                          MathAbs(g_mtf_cache[last_idx].negative);
      g_mtf_cache[last_idx].abs_diff = MathAbs(g_mtf_cache[last_idx].difference);
      
      // Recalculer les MA pour les dernières barres
      UpdateLastMAs();
      
      return true;
   }
   
   // Si la direction a changé, on doit reconstruire
   return false;
}

//+------------------------------------------------------------------+
//| Mettre à jour les MA pour les dernières barres                   |
//+------------------------------------------------------------------+
void UpdateLastMAs()
{
   if(g_mtf_cache_count < 2) return;
   
   int last_idx = g_mtf_cache_count - 1;
   
   // MA Différence
   if(last_idx >= InpMAPeriod - 1)
   {
      if(InpMAMethod == MODE_EMA)
      {
         double alpha = 2.0 / (InpMAPeriod + 1.0);
         g_mtf_cache[last_idx].ma_diff = alpha * g_mtf_cache[last_idx].difference + 
                                         (1.0 - alpha) * g_mtf_cache[last_idx-1].ma_diff;
      }
      else
      {
         double sum = 0.0;
         for(int j = 0; j < InpMAPeriod; j++)
            sum += g_mtf_cache[last_idx - j].difference;
         g_mtf_cache[last_idx].ma_diff = sum / InpMAPeriod;
      }
   }
   
   // MA Abs
   if(last_idx >= InpMAAbsPeriod - 1)
   {
      if(InpMAAbsMethod == MODE_EMA)
      {
         double alpha = 2.0 / (InpMAAbsPeriod + 1.0);
         g_mtf_cache[last_idx].ma_abs = alpha * g_mtf_cache[last_idx].abs_diff + 
                                        (1.0 - alpha) * g_mtf_cache[last_idx-1].ma_abs;
      }
      else
      {
         double sum = 0.0;
         for(int j = 0; j < InpMAAbsPeriod; j++)
            sum += g_mtf_cache[last_idx - j].abs_diff;
         g_mtf_cache[last_idx].ma_abs = sum / InpMAAbsPeriod;
      }
   }
}

//+------------------------------------------------------------------+
//| Calculer les MA sur le cache                                     |
//+------------------------------------------------------------------+
void CalculateMAsOnCache()
{
   if(g_mtf_cache_count < 2) return;
   
   double alpha_diff = 2.0 / (InpMAPeriod + 1.0);
   double alpha_abs = 2.0 / (InpMAAbsPeriod + 1.0);
   
   for(int i = 0; i < g_mtf_cache_count; i++)
   {
      // MA Différence
      if(i < InpMAPeriod - 1)
      {
         g_mtf_cache[i].ma_diff = 0.0;
      }
      else if(InpMAMethod == MODE_EMA)
      {
         if(i == InpMAPeriod - 1)
         {
            double sum = 0.0;
            for(int j = 0; j < InpMAPeriod; j++)
               sum += g_mtf_cache[i - j].difference;
            g_mtf_cache[i].ma_diff = sum / InpMAPeriod;
         }
         else
         {
            g_mtf_cache[i].ma_diff = alpha_diff * g_mtf_cache[i].difference + 
                                     (1.0 - alpha_diff) * g_mtf_cache[i-1].ma_diff;
         }
      }
      else if(InpMAMethod == MODE_SMMA)
      {
         if(i == InpMAPeriod - 1)
         {
            double sum = 0.0;
            for(int j = 0; j < InpMAPeriod; j++)
               sum += g_mtf_cache[i - j].difference;
            g_mtf_cache[i].ma_diff = sum / InpMAPeriod;
         }
         else if(i > InpMAPeriod - 1)
         {
            g_mtf_cache[i].ma_diff = (g_mtf_cache[i-1].ma_diff * (InpMAPeriod - 1) + 
                                      g_mtf_cache[i].difference) / InpMAPeriod;
         }
      }
      else if(InpMAMethod == MODE_LWMA)
      {
         double weighted_sum = 0.0;
         int weight_sum = 0;
         for(int j = 0; j < InpMAPeriod && (i - j) >= 0; j++)
         {
            int weight = InpMAPeriod - j;
            weighted_sum += g_mtf_cache[i - j].difference * weight;
            weight_sum += weight;
         }
         if(weight_sum > 0)
            g_mtf_cache[i].ma_diff = weighted_sum / weight_sum;
      }
      else // SMA
      {
         double sum = 0.0;
         int count = 0;
         for(int j = 0; j < InpMAPeriod && (i - j) >= 0; j++)
         {
            sum += g_mtf_cache[i - j].difference;
            count++;
         }
         if(count > 0)
            g_mtf_cache[i].ma_diff = sum / count;
      }
      
      // MA Abs Diff
      if(i < InpMAAbsPeriod - 1)
      {
         g_mtf_cache[i].ma_abs = 0.0;
      }
      else if(InpMAAbsMethod == MODE_EMA)
      {
         if(i == InpMAAbsPeriod - 1)
         {
            double sum = 0.0;
            for(int j = 0; j < InpMAAbsPeriod; j++)
               sum += g_mtf_cache[i - j].abs_diff;
            g_mtf_cache[i].ma_abs = sum / InpMAAbsPeriod;
         }
         else
         {
            g_mtf_cache[i].ma_abs = alpha_abs * g_mtf_cache[i].abs_diff + 
                                    (1.0 - alpha_abs) * g_mtf_cache[i-1].ma_abs;
         }
      }
      else if(InpMAAbsMethod == MODE_SMMA)
      {
         if(i == InpMAAbsPeriod - 1)
         {
            double sum = 0.0;
            for(int j = 0; j < InpMAAbsPeriod; j++)
               sum += g_mtf_cache[i - j].abs_diff;
            g_mtf_cache[i].ma_abs = sum / InpMAAbsPeriod;
         }
         else if(i > InpMAAbsPeriod - 1)
         {
            g_mtf_cache[i].ma_abs = (g_mtf_cache[i-1].ma_abs * (InpMAAbsPeriod - 1) + 
                                     g_mtf_cache[i].abs_diff) / InpMAAbsPeriod;
         }
      }
      else if(InpMAAbsMethod == MODE_LWMA)
      {
         double weighted_sum = 0.0;
         int weight_sum = 0;
         for(int j = 0; j < InpMAAbsPeriod && (i - j) >= 0; j++)
         {
            int weight = InpMAAbsPeriod - j;
            weighted_sum += g_mtf_cache[i - j].abs_diff * weight;
            weight_sum += weight;
         }
         if(weight_sum > 0)
            g_mtf_cache[i].ma_abs = weighted_sum / weight_sum;
      }
      else // SMA
      {
         double sum = 0.0;
         int count = 0;
         for(int j = 0; j < InpMAAbsPeriod && (i - j) >= 0; j++)
         {
            sum += g_mtf_cache[i - j].abs_diff;
            count++;
         }
         if(count > 0)
            g_mtf_cache[i].ma_abs = sum / count;
      }
   }
}

//+------------------------------------------------------------------+
//| Recherche binaire dans le cache MTF                              |
//+------------------------------------------------------------------+
int FindMTFBarIndex(datetime target_time)
{
   if(g_mtf_cache_count == 0) return -1;
   
   int left = 0;
   int right = g_mtf_cache_count - 1;
   int result = -1;
   
   while(left <= right)
   {
      int mid = (left + right) / 2;
      
      if(g_mtf_cache[mid].time <= target_time)
      {
         result = mid;
         left = mid + 1;
      }
      else
      {
         right = mid - 1;
      }
   }
   
   return result;
}

//+------------------------------------------------------------------+
//| Appliquer le lissage EMA sur un buffer                           |
//+------------------------------------------------------------------+
void ApplySmoothing(double &buffer[], int start, int end, int period)
{
   if(period <= 1 || start >= end) return;
   
   double alpha = 2.0 / (period + 1.0);
   
   // Premier passage : calculer l'EMA
   double ema = buffer[start];
   
   for(int i = start + 1; i < end; i++)
   {
      ema = alpha * buffer[i] + (1.0 - alpha) * ema;
      buffer[i] = ema;
   }
}

//+------------------------------------------------------------------+
//| Appliquer le lissage sur tous les buffers                        |
//+------------------------------------------------------------------+
void ApplySmoothingToAllBuffers(int start, int end)
{
   if(g_smoothing_period <= 1) return;
   
   ApplySmoothing(PositiveBuffer, start, end, g_smoothing_period);
   ApplySmoothing(NegativeBuffer, start, end, g_smoothing_period);
   ApplySmoothing(DifferenceBuffer, start, end, g_smoothing_period);
   ApplySmoothing(AbsDifferenceBuffer, start, end, g_smoothing_period);
   ApplySmoothing(MADifferenceBuffer, start, end, g_smoothing_period);
   ApplySmoothing(MAAbsDifferenceBuffer, start, end, g_smoothing_period);
   
   // Recalculer l'inverse après lissage
   for(int i = start; i < end; i++)
   {
      InverseMAAbsDifferenceBuffer[i] = -MAAbsDifferenceBuffer[i];
   }
}

//+------------------------------------------------------------------+
//| OnCalculate                                                      |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
   if(rates_total < 10) return 0;
   
   // Déterminer si on doit reconstruire le cache
   int current_mtf_bars = iBars(_Symbol, g_timeframe);
   bool need_rebuild = false;
   
   if(g_mtf_cache_count == 0)
   {
      need_rebuild = true;
   }
   else if(current_mtf_bars != g_mtf_last_bars_count)
   {
      // Nouvelle barre MTF
      need_rebuild = true;
   }
   else
   {
      // Même nombre de barres, essayer de mettre à jour seulement la dernière
      if(!UpdateLastMTFBar())
         need_rebuild = true;
   }
   
   if(need_rebuild)
   {
      if(!RebuildMTFCache())
      {
         if(InpDebugMode)
            Print("DEBUG: Échec reconstruction cache MTF");
         return 0;
      }
   }
   
   if(g_mtf_cache_count == 0) return 0;
   
   // Déterminer la plage à calculer
   int start = (prev_calculated == 0) ? 0 : prev_calculated - 1;
   
   // Mapper les données MTF vers le timeframe courant
   for(int i = start; i < rates_total; i++)
   {
      int mtf_idx = FindMTFBarIndex(time[i]);
      
      if(mtf_idx >= 0 && mtf_idx < g_mtf_cache_count)
      {
         PositiveBuffer[i] = g_mtf_cache[mtf_idx].positive;
         NegativeBuffer[i] = g_mtf_cache[mtf_idx].negative;
         DifferenceBuffer[i] = g_mtf_cache[mtf_idx].difference;
         AbsDifferenceBuffer[i] = g_mtf_cache[mtf_idx].abs_diff;
         MADifferenceBuffer[i] = g_mtf_cache[mtf_idx].ma_diff;
         MAAbsDifferenceBuffer[i] = g_mtf_cache[mtf_idx].ma_abs;
         InverseMAAbsDifferenceBuffer[i] = -g_mtf_cache[mtf_idx].ma_abs;
      }
      else
      {
         PositiveBuffer[i] = 0.0;
         NegativeBuffer[i] = 0.0;
         DifferenceBuffer[i] = 0.0;
         AbsDifferenceBuffer[i] = 0.0;
         MADifferenceBuffer[i] = 0.0;
         MAAbsDifferenceBuffer[i] = 0.0;
         InverseMAAbsDifferenceBuffer[i] = 0.0;
      }
   }
   
   // Appliquer le lissage si activé et en mode MTF
   if(InpEnableMTFSmoothing && g_smoothing_period > 1)
   {
      ApplySmoothingToAllBuffers(start, rates_total);
   }
   
   // Alertes
   if(prev_calculated > 0 && InpEnableAlerts && rates_total >= 2)
   {
      int last = rates_total - 1;
      
      if(InpAlertOnSeriesStart)
      {
         if(PositiveBuffer[last] > 0 && PositiveBuffer[last-1] == 0)
            Alert(_Symbol, " - MTF ", EnumToString(g_timeframe), " - Démarrage série HAUSSIÈRE");
         if(NegativeBuffer[last] < 0 && NegativeBuffer[last-1] == 0)
            Alert(_Symbol, " - MTF ", EnumToString(g_timeframe), " - Démarrage série BAISSIÈRE");
      }
      
      if(InpAlertOnSeriesStop)
      {
         if(PositiveBuffer[last] == 0 && PositiveBuffer[last-1] > 0)
            Alert(_Symbol, " - MTF ", EnumToString(g_timeframe), " - Arrêt série HAUSSIÈRE");
         if(NegativeBuffer[last] == 0 && NegativeBuffer[last-1] < 0)
            Alert(_Symbol, " - MTF ", EnumToString(g_timeframe), " - Arrêt série BAISSIÈRE");
      }
      
      if(InpAlertOnCrossover && last > 1)
      {
         if(MADifferenceBuffer[last] > 0 && MADifferenceBuffer[last-1] <= 0)
            Alert(_Symbol, " - MTF ", EnumToString(g_timeframe), " - Croisement HAUSSIER MA");
         else if(MADifferenceBuffer[last] < 0 && MADifferenceBuffer[last-1] >= 0)
            Alert(_Symbol, " - MTF ", EnumToString(g_timeframe), " - Croisement BAISSIER MA");
      }
   }
   
   return rates_total;
}

//+------------------------------------------------------------------+
//| Deinitialization                                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   if(g_atr_handle != INVALID_HANDLE)
      IndicatorRelease(g_atr_handle);
   
   ArrayFree(g_mtf_cache);
   
   Print("=== HEIKIN ASHI CUMULATOR MTF OPTIMIZED DÉCHARGÉ ===");
}
//+------------------------------------------------------------------+
